#!/usr/bin/perl -w
use strict;
use HTTP::Request::Common;
use LWP::UserAgent;
use Digest::MD5;
use File::Temp qw/tempfile/;
use Net::Ping;
use YAML;
use URI;

#customize these if needed
my $home = $ENV{'HOME'};
my $tempdir = "$home/jobs";
my $me = "$home/bin/ljcomm.pl";

# post LJ comment replies
# invoke with the last argument 
# ljreply-$journal-$entry-$comment@livejournal.com.
# if --canfail is supplied before the ljreply argument,
# then the script may fail and do nothing, otherwise if it
# fails it will write itself out as a delayed job.

my ($headers, $body);
my $inheaders = 1;
my $toline;
my $ctline;
my @lines;
my $sendmail = 0; # when we are sendmail
my $status;

my $to = $ARGV[-1] || "";
my $canfail = ($ARGV[-2] || "") eq '--canfail';

$status = "[bad argument]", out(1)
    unless my ($journal, $entry, $parent) =
           $to =~ /^ljreply-(\w+)-(\d+)-(\d+)\@livejournal.com$/;

sub out {
    my $code = shift;
    print "$status\n";
    # if we can't fail, we must delay
    exit $code if $canfail;
    exit $code if $code == 0;
    print "[delayed: " . delay() . "]\n";
    exit 0;
}

sub delay {
    my ($fh,$name) = tempfile("ljcommXXXXXX", DIR => $tempdir);
    print $fh <<END;
#!/bin/bash
$me --canfail $to <<'DELIM-$name'
END
    for (@lines) { print $fh $_; }
    print $fh "DELIM-$name\n";
    print $fh <<END;
if [ "\$?" -eq "0" ]; then
  cp \$0 /tmp/
  echo "removing \$0"
  rm \$0
  exit 0
else
  exit 1
fi
END
    close $fh;
    chmod 0755, $name;
    return $name;
}

sub net {
    my $p = Net::Ping->new();
    return $p->ping("www.livejournal.com");
}
 
@lines = <STDIN>;

$status = "[no net]", out(1)
    unless net();

foreach (@lines) {
  $inheaders = 0 if $_ eq "\n";
  
  if ($inheaders) {
    $headers .= $_;
    if (m/^Content-Type:/) {
      $ctline = $_;
    }
  } else {
    $body .= $_;
  }
}

# store a backup copy
open O, ">>$home/ljreply-backup.txt";
print O "To: $to\nBody:\n\n$body\n\n";
close O;

# lose the sig in the body, if any
$body =~ s/^--(\s)?\n.*//sm;

# also consider a $ sign alone on the line as a stop sign
$body =~ s/^\$\n.*//sm;

# delete the "You wrote on..." line if it comes first; then,
# if it's still present, delete it and everything after it,
# because that's a top-style reply, and the line starts the
# needless copy of the quoted parent.
$body =~ s/^You wrote on .*//; 
$body =~ s/^You wrote on .*//sm;
 
# lose initial and final newline(s) in the body
$body =~ s/^(\n)+//s;
$body =~ s/(\n)+$//s;

# Handle quoting
# replace every "> X" with "<i>X</i>", then 
# replace adjacent </i>\n<i> with a space. That should do the trick.

$body =~ s!^> (.*)$!<i>$1</i>!mg; 
$body =~ s!</i>\n<i>! !sg;

# unwrap: replace every \n which is followed by non-whitespace
# with a space. This allows to e.g. quote verse w/o unwrapping by
# putting an initial space on those lines. Also demand that the \n
# is not preceded by another \n or a period.
$body =~ s!([^\n\.])\n(\S)!$1 $2!sg;

# delete unnecessary newlines after <i> and before </i>
$body =~ s!<i>(\n)+(?=\S)!<i>!sg;
$body =~ s!(\n)+</i>!</i>!sg;

my $ua = LWP::UserAgent->new;

my $uri = "http://www.livejournal.com/talkpost_do.bml";

my ($user, $pwd) = ("avva", "yeah_right");

my $encoding = "";
if ($ctline =~ m!charset=(\S*)$!) {
  $encoding = $1;
}

# don't send utf-8 as an encoding, just send an empty string then
# otherwise MapToUTF8 in LJ code gets confused
if ($encoding eq 'utf-8') {
  $encoding = '';
}

# note parenttalkid used in the form and to calculate ecphash
# is private, not public.
my $parentid = int($parent / 256);

# calculate ecphash
# private itemid is used
my $ecphash = "ecph-" . Digest::MD5::md5_hex(int($entry/256) . $parentid . $pwd);

# build the request
# temporarily use GET. POST support is broken on server (wants form auth).
# as of 2006/03.

my $uri_object = URI->new($uri);

my $fields = [
		      parenttalkid => $parentid,
		      journal      => $journal,
		      itemid       => $entry,
		      usertype     => 'user',
		      userpost     => $user,
		      ecphash      => $ecphash,
		      encoding     => $encoding,
		      # encoding
		      # subject
		      body         => $body,
		      submit       => "Post Reply",
		     ];

my $req = POST $uri, $fields;
#$uri_object->query_form(@$fields);
#my $req = GET $uri_object;
my $res = $ua->request($req);

open O, ">>$home/test.txt";
print O "URL of parent: http://www.livejournal.com/users/$journal/$entry.html?thread=$parent\n";
print O $to;
print O "encoding: $encoding\n";
print O "ctline: $ctline";
print O "request: " . Dump($req);

if ($res->is_error()) {
  print O "HTTP ERROR!\n";
  print O "code: " . $res->code;
  $status = "[error: " . $res->code . "]";
  out(1);
}

print O "HTTP response:\n";
print O $res->content;
close O;

my $content = $res->content;

if ($content =~ m!<title>Error</title>! ||
    $content =~ m!<title>Comment Not Posted</title>!) {
  $status = "[error: server-side]", out(1);
}
exit 0;
